package com.inzy.wsmock

import io.netty.channel.ChannelHandlerContext
import io.netty.channel.SimpleChannelInboundHandler
import io.netty.handler.codec.http.FullHttpRequest
import io.netty.handler.codec.http.websocketx.TextWebSocketFrame
import io.netty.util.AttributeKey

class RequestParamHandler : SimpleChannelInboundHandler<TextWebSocketFrame?>() {

    // 定义存储查询参数的AttributeKey（全局复用）
    companion object {
        val REQUEST_PARAMS_KEY = AttributeKey.valueOf<Map<String, String>>("REQUEST_PARAMS")
        val REQUEST_PATH_KEY = AttributeKey.valueOf<String>("REQUEST_PATH") // 存储请求路径

        val PARAM_TYPE_KEY = "type"
        val PARAM_TYPE_VALUE_1 = "1"
        val PARAM_TYPE_VALUE_2 = "2"
        val PARAM_TYPE_VALUE_3 = "3"
        val PARAM_TYPE_VALUE_4 = "4"
    }

    @Throws(Exception::class)
    override fun channelRead(ctx: ChannelHandlerContext, msg: Any) {
        if (msg is FullHttpRequest) {
            val request = msg
            val uri = request.uri()

            // 1. 解析请求路径和查询参数
            val (path, params) = parseUri(uri)

            // 2. 打印调试信息
//            println("请求路径: $path")
//            println("查询参数: $params")

            // 3. 将路径和参数存入Channel属性（供后续处理器使用）
            ctx.channel().attr(REQUEST_PATH_KEY).set(path)
            ctx.channel().attr(REQUEST_PARAMS_KEY).set(params)

            // 4. 可选：去除查询参数后更新请求URI（不影响后续处理器解析路径）
            request.uri = path.substring(0..2)

            // 去除查询参数后更新URI（可选）
            /*   if (uri.contains("?")) {
                   val newUri = uri.substring(0, uri.indexOf("?"))
                   request.setUri(newUri)
               }*/
        }
        // 处理WebSocket消息
        super.channelRead(ctx, msg)
    }

    @Throws(Exception::class)
    override fun channelRead0(ctx: ChannelHandlerContext?, msg: TextWebSocketFrame?) {
    }


    // 核心方法：解析URI，返回 Pair(路径, 参数字典)
    private fun parseUri(uri: String): Pair<String, Map<String, String>> {
        // 分割路径和参数（? 是分隔符）
        val splitIndex = uri.indexOf("?")
        return if (splitIndex == -1) {
            // 无查询参数的情况
            Pair(uri, emptyMap())
        } else {
            // 有查询参数的情况：解析路径 + 解析参数
            val path = uri.substring(0, splitIndex)
            val paramStr = uri.substring(splitIndex + 1)
            val params = parseParams(paramStr)
            Pair(path, params)
        }
    }

    // 解析查询参数字符串（a=1&b=2 形式）
    private fun parseParams(paramStr: String): Map<String, String> {
        val params = mutableMapOf<String, String>()
        if (paramStr.isBlank()) return params

        // 按 & 分割多个参数
        val paramArray = paramStr.split("&").filter { it.isNotBlank() }
        for (param in paramArray) {
            // 按 = 分割参数名和值
            val keyValue = param.split("=", limit = 2) // limit=2 避免值包含=的情况
            if (keyValue.size == 2) {
                val key = keyValue[0].trim()
                val value = keyValue[1].trim()
                params[key] = value
            }
        }
        return params
    }


    // 自定义方法：解析URL参数
    /*  private fun getUrlParams(url: String): MutableMap<String?, String?> {
          var url = url
          val map: MutableMap<String?, String?> = HashMap<String?, String?>()
          url = url.replace("?", ";")
          if (!url.contains(";")) return map
          val arr = url.split(";".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray()[1].split("&".toRegex())
              .dropLastWhile { it.isEmpty() }.toTypedArray()
          for (s in arr) {
              val pair = s.split("=".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray()
              if (pair.size == 2) {
                  map.put(pair[0], pair[1])
              }
          }
          return map
      }*/
}
