package com.inzy.wsmock


import io.netty.bootstrap.ServerBootstrap
import io.netty.channel.ChannelInitializer
import io.netty.channel.ChannelOption
import io.netty.channel.nio.NioEventLoopGroup
import io.netty.channel.socket.SocketChannel
import io.netty.channel.socket.nio.NioServerSocketChannel
import io.netty.handler.codec.http.HttpObjectAggregator
import io.netty.handler.codec.http.HttpServerCodec
import io.netty.handler.codec.http.websocketx.WebSocketServerProtocolHandler
import io.netty.handler.stream.ChunkedWriteHandler
import jakarta.annotation.PostConstruct
import jakarta.annotation.PreDestroy
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Value
import org.springframework.stereotype.Component

@Component
class NettyWebSocketServer(
    @Value("\${netty.websocket.port:8089}") private val port: Int,
    private val webSocketHandler: WebSocketHandler
) {
    private val logger = LoggerFactory.getLogger(javaClass)

    // Netty主从线程组
    private val bossGroup = NioEventLoopGroup(1)
    private val workerGroup = NioEventLoopGroup()

    // 保存Netty服务的ChannelFuture，用于优雅关闭
    private var serverChannelFuture: io.netty.channel.ChannelFuture? = null

    /**
     * 启动Netty WebSocket服务（异步执行，避免阻塞Spring主线程）
     */
    @PostConstruct
    fun start() {
        // 🔥 关键修改：使用新线程启动Netty，不阻塞Spring初始化
        CoroutineScope(Dispatchers.IO).launch {
            try {
                val bootstrap = ServerBootstrap()
                bootstrap.group(bossGroup, workerGroup)
                    .channel(NioServerSocketChannel::class.java)
                    .option(ChannelOption.SO_BACKLOG, 1024)
                    .childOption(ChannelOption.SO_KEEPALIVE, true)
                    .childHandler(object : ChannelInitializer<SocketChannel>() {
                        override fun initChannel(ch: SocketChannel) {
                            val pipeline = ch.pipeline()
                            pipeline.addLast(HttpServerCodec())
                            pipeline.addLast(ChunkedWriteHandler())
                            pipeline.addLast(HttpObjectAggregator(1024 * 1024))
                            pipeline.addLast(WebSocketServerProtocolHandler("/gs-guide-websocket"))
                            pipeline.addLast(webSocketHandler)
                        }
                    })
                // 绑定端口并启动（非阻塞？不，这里sync()只阻塞当前新线程，不阻塞Spring主线程）
                serverChannelFuture = bootstrap.bind(port).sync()
                logger.info("Netty WebSocket服务启动成功，端口：$port")

                // 阻塞当前新线程（而非Spring主线程）直到服务关闭
                serverChannelFuture?.channel()?.closeFuture()?.sync()
            } catch (e: Exception) {
                logger.error("Netty服务启动失败", e)
            } finally {
                // 优雅关闭线程组
                bossGroup.shutdownGracefully()
                workerGroup.shutdownGracefully()
                logger.info("Netty线程组已关闭")
            }
        }
    }

    /**
     * 销毁时关闭Netty服务
     */
    @PreDestroy
    fun stop() {
        // 主动关闭Netty服务，解除closeFuture的阻塞
        serverChannelFuture?.channel()?.close()?.sync()
        bossGroup.shutdownGracefully()
        workerGroup.shutdownGracefully()
        logger.info("Netty WebSocket服务已关闭")
    }
}