package com.sd.cavphmi;

import com.sd.cavphmi.utils.MyMapUtils;

import org.locationtech.proj4j.*;
import java.util.ArrayList;
import java.util.List;

public class CoordinateProjectionUtils {
    private static final CRSFactory crsFactory = new CRSFactory();

    // 定义坐标系参数
    private static final String WGS84_PARAMS = "+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs";
    private static final String CGCS2000_PARAMS = "+proj=tmerc +lat_0=0 +lon_0=116 +k=1 +x_0=500000 +y_0=0 +ellps=GRS80 +units=m +no_defs";
    // 北京主要使用的3度分带投影
    public static final String CGCS2000_BEIJING_3_DEGREE_ZONE_39 =
            "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=500000 +y_0=0 +ellps=GRS80 +units=m +no_defs";
    // CGCS2000 常用投影参数
    public static final String CGCS2000_GEOGRAPHIC = "+proj=longlat +ellps=GRS80 +no_defs";
    public static final String CGCS2000_3_DEGREE_ZONE_37 = "+proj=tmerc +lat_0=0 +lon_0=111 +k=1 +x_0=37500000 +y_0=0 +ellps=GRS80 +units=m +no_defs";
    public static final String CGCS2000_3_DEGREE_ZONE_38 = "+proj=tmerc +lat_0=0 +lon_0=114 +k=1 +x_0=38500000 +y_0=0 +ellps=GRS80 +units=m +no_defs";
    public static final String CGCS2000_3_DEGREE_ZONE_39 = "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=39500000 +y_0=0 +ellps=GRS80 +units=m +no_defs";
    public static final String CGCS2000_3_DEGREE_ZONE_40 = "+proj=tmerc +lat_0=0 +lon_0=120 +k=1 +x_0=40500000 +y_0=0 +ellps=GRS80 +units=m +no_defs";

    // 6度带投影参数
    public static final String CGCS2000_6_DEGREE_ZONE_13 = "+proj=tmerc +lat_0=0 +lon_0=75 +k=1 +x_0=13500000 +y_0=0 +ellps=GRS80 +units=m +no_defs";
    public static final String CGCS2000_6_DEGREE_ZONE_20 = "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=20500000 +y_0=0 +ellps=GRS80 +units=m +no_defs";
    public static final String CGCS2000_6_DEGREE_ZONE_21 = "+proj=tmerc +lat_0=0 +lon_0=123 +k=1 +x_0=21500000 +y_0=0 +ellps=GRS80 +units=m +no_defs";


    /**
     * 计算点到坐标点串的投影距离
     * @param point 待投影点 [经度, 纬度]
     * @param coordinateSeries 坐标点串 [[经度1, 纬度1], [经度2, 纬度2], ...]
     * @return 投影结果数组 [投影点经度, 投影点纬度, 最近点索引, 最小距离]
     */
    public static double[] calculatePointProjection(double[] point, List<double[]> coordinateSeries) {
        try {
            // 创建坐标系
            CoordinateReferenceSystem wgs84 = crsFactory.createFromParameters("WGS84", WGS84_PARAMS);
            CoordinateReferenceSystem cgcs2000 = crsFactory.createFromParameters("CGCS2000", CGCS2000_BEIJING_3_DEGREE_ZONE_39);
            CoordinateTransform transform = new CoordinateTransformFactory().createTransform(wgs84, cgcs2000);

            // 转换待投影点到CGCS2000坐标系
//            ProjCoordinate srcCoord = new ProjCoordinate(point[0], point[1]);
//            ProjCoordinate dstCoord = new ProjCoordinate();
//            transform.transform(srcCoord, dstCoord);
//            double[] projectedPoint = {dstCoord.x, dstCoord.y};

            //直接使用02坐标系
            ProjCoordinate dstCoord = new ProjCoordinate(point[0], point[1]);
            double[] projectedPoint = {dstCoord.x, dstCoord.y};

            // 转换坐标点串到CGCS2000坐标系
            List<double[]> projectedSeries = new ArrayList<>();
            for (double[] coord : coordinateSeries) {
//                ProjCoordinate src = new ProjCoordinate(coord[0], coord[1]);
//                ProjCoordinate dst = new ProjCoordinate();
//                transform.transform(src, dst);
//                projectedSeries.add(new double[]{dst.x, dst.y});
                projectedSeries.add(new double[]{coord[0], coord[1]});
            }

            // 计算点到线串的最短距离和投影点
            return findNearestProjection(projectedPoint, projectedSeries);

        } catch (Proj4jException e) {
            e.printStackTrace();
            return new double[]{0, 0, -1, -1};
        }
    }

    /**
     * 在投影后的坐标系中查找最近投影点
     */
    private static double[] findNearestProjection(double[] point, List<double[]> series) {
        double minDistance = Double.MAX_VALUE;
        int nearestIndex = -1;
        double[] projectionPoint = new double[2];

        // 遍历所有线段计算投影
        for (int i = 0; i < series.size() - 1; i++) {
            double[] start = series.get(i);
            double[] end = series.get(i + 1);

            double[] currentProjection = projectPointToLine(point, start, end);
            double distance = calculateDistance(point, currentProjection);

            if (distance < minDistance) {
                minDistance = distance;
                nearestIndex = i;
                projectionPoint = currentProjection;
            }
        }

        return new double[]{
                projectionPoint[0],
                projectionPoint[1],
                nearestIndex,
                minDistance
        };
    }

    /**
     * 计算点到线段的投影
     */
    private static double[] projectPointToLine(double[] point, double[] lineStart, double[] lineEnd) {
        double ax = point[0] - lineStart[0];
        double ay = point[1] - lineStart[1];
        double bx = lineEnd[0] - lineStart[0];
        double by = lineEnd[1] - lineStart[1];

        double dot = ax * bx + ay * by;
        double lenSq = bx * bx + by * by;

        double t = (lenSq != 0) ? Math.max(0, Math.min(1, dot / lenSq)) : 0;

        return new double[]{
                lineStart[0] + t * bx,
                lineStart[1] + t * by
        };
    }

    // 地球半径（米），WGS84/CGCS2000 椭球近似半径
    private static final double EARTH_RADIUS = 6371008.8;
    /**
     * 计算两点间距离
     */
    private static double calculateDistance(double[] p1, double[] p2) {
//        double dx = p1[0] - p2[0];
//        double dy = p1[1] - p2[1];
//        return Math.sqrt(dx * dx + dy * dy);
        // 角度转弧度
        double radLat1 = Math.toRadians(p1[1]);
        double radLon1 = Math.toRadians(p1[0]);
        double radLat2 = Math.toRadians(p2[1]);
        double radLon2 = Math.toRadians(p2[0]);

        // 纬度差、经度差
        double deltaLat = radLat2 - radLat1;
        double deltaLon = radLon2 - radLon1;

        // Haversine 公式
        double a = Math.sin(deltaLat / 2) * Math.sin(deltaLat / 2)
                + Math.cos(radLat1) * Math.cos(radLat2)
                * Math.sin(deltaLon / 2) * Math.sin(deltaLon / 2);
        double c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));

        // 距离 = 地球半径 × 圆心角
        return EARTH_RADIUS * c;
    }

    /**
     * 示例用法
     */
    public static void main(String[] args) {
        // 待投影点 (WGS84坐标系)
        double[] testPoint = {116.3974, 39.9093}; // 北京天安门

        // 坐标点串 (WGS84坐标系)
        List<double[]> coordinateSeries = new ArrayList<>();
        coordinateSeries.add(new double[]{116.3914, 39.9053});
        coordinateSeries.add(new double[]{116.4034, 39.9133});
        coordinateSeries.add(new double[]{116.4154, 39.9213});

        // 计算投影
        double[] result = calculatePointProjection(testPoint, coordinateSeries);

        System.out.println("投影点坐标: (" + result[0] + ", " + result[1] + ")");
        System.out.println("最近线段索引: " + (int)result[2]);
        System.out.println("最小距离: " + result[3] + " 米");
    }
}
