package com.sd.cavphmi.utils

import org.locationtech.proj4j.CRSFactory
import org.locationtech.proj4j.Proj4jException
import org.locationtech.proj4j.ProjCoordinate
import kotlin.math.atan2
import kotlin.math.cos
import kotlin.math.max
import kotlin.math.min
import kotlin.math.sin
import kotlin.math.sqrt

/**坐标投影转换工具***/
object Proj4jCoord {
    private val crsFactory: CRSFactory = CRSFactory()
    //返回结果
    private val resultArray=doubleArrayOf(0.0, 0.0, 0.0, 0.0)

    // 定义坐标系参数
    private const val WGS84_PARAMS = "+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs"
    private const val CGCS2000_PARAMS =
        "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=500000 +y_0=0 +ellps=GRS80 +units=m +no_defs"

    const val CGCS2000_BEIJING_3_DEGREE_ZONE_39: String =
        "+proj=tmerc +lat_0=0 +lon_0=117 +k=1 +x_0=500000 +y_0=0 +ellps=GRS80 +units=m +no_defs"

    /**
     * 计算点到坐标点串的投影距离
     * @param point 待投影点 [经度, 纬度]
     * @param coordinateSeries 坐标点串 [[经度1, 纬度1], [经度2, 纬度2], ...]
     * @return 投影结果数组 [投影点经度, 投影点纬度, 最近点索引, 最小距离]
     */
    fun calculatePointProjection(
        point: DoubleArray,
        coordinateSeries: MutableList<DoubleArray>
    ): DoubleArray {
        try {
            // 创建坐标系
            /*    val wgs84: CoordinateReferenceSystem? =
                    crsFactory.createFromParameters(
                        "WGS84",
                        WGS84_PARAMS
                    )
                val cgcs2000: CoordinateReferenceSystem? =
                    crsFactory.createFromParameters(
                        "CGCS2000",
                        CGCS2000_BEIJING_3_DEGREE_ZONE_39
                    )*/

            //直接使用02坐标系
            val dstCoord = ProjCoordinate(point[0], point[1])
            val projectedPoint = doubleArrayOf(dstCoord.x, dstCoord.y)

            // 转换坐标点串到CGCS2000坐标系
//            var projectedSeries = mutableListOf<DoubleArray>()
//            for (coord in coordinateSeries) {
//                projectedSeries.add(doubleArrayOf(coord[0], coord[1]))
//            }

            // 计算点到线串的最短距离和投影点
            return findNearestProjection(projectedPoint, coordinateSeries)
        } catch (e: Proj4jException) {
            e.printStackTrace()
            return doubleArrayOf(0.0, 0.0, -1.0, -1.0)
        }
    }


    /**
     * 在投影后的坐标系中查找最近投影点
     */
    private fun findNearestProjection(
        point: DoubleArray,
        series: MutableList<DoubleArray>
    ): DoubleArray {
        var minDistance = Double.Companion.MAX_VALUE
        var nearestIndex = -1
        var projectionPoint = DoubleArray(2)

        // 遍历所有线段计算投影
        for (i in 0..<series.size - 1) {
            val start = series.get(i)
            val end = series.get(i + 1)

            val currentProjection: DoubleArray = projectPointToLine(point, start, end)
//            val currentProjection: DoubleArray = series.get(i)
            val distance: Double = calculateDistance(point, currentProjection)

            if (distance < minDistance) {
                minDistance = distance
                nearestIndex = i
                projectionPoint = currentProjection
            }
        }
        resultArray.set(0,projectionPoint[0])
        resultArray.set(1,projectionPoint[1])
        resultArray.set(2,nearestIndex.toDouble())
        resultArray.set(3,minDistance)

        return resultArray

//        return doubleArrayOf(
//            projectionPoint[0],
//            projectionPoint[1],
//            nearestIndex.toDouble(),
//            minDistance
//        )
    }


    /**
     * 计算点到线段的投影
     */
    private fun projectPointToLine(
        point: DoubleArray,
        lineStart: DoubleArray,
        lineEnd: DoubleArray
    ): DoubleArray {
        val ax = point[0] - lineStart[0]
        val ay = point[1] - lineStart[1]
        val bx = lineEnd[0] - lineStart[0]
        val by = lineEnd[1] - lineStart[1]

        val dot = ax * bx + ay * by
        val lenSq = bx * bx + by * by

        val t = if (lenSq != 0.0) max(0.0, min(1.0, dot / lenSq)) else 0.0

        return doubleArrayOf(
            lineStart[0] + t * bx,
            lineStart[1] + t * by
        )
    }

    // 地球半径（米），WGS84/CGCS2000 椭球近似半径
    private const val EARTH_RADIUS: Double = 6371008.8

    /**
     * 计算两点间距离
     */
    private fun calculateDistance(p1: DoubleArray, p2: DoubleArray): Double {
//        var dx = p1[0] - p2[0]
//        var dy = p1[1] - p2[1]
//        return Math.sqrt(dx * dx + dy * dy)

        // 角度转弧度
        val radLat1 = Math.toRadians(p1[1])
        val radLon1 = Math.toRadians(p1[0])
        val radLat2 = Math.toRadians(p2[1])
        val radLon2 = Math.toRadians(p2[0])

        // 纬度差、经度差
        val deltaLat = radLat2 - radLat1
        val deltaLon = radLon2 - radLon1

        // Haversine 公式
        val a = (sin(deltaLat / 2) * sin(deltaLat / 2)
                + cos(radLat1) * cos(radLat2) * sin(deltaLon / 2) * sin(deltaLon / 2))
        val c = 2 * atan2(sqrt(a), sqrt(1 - a))

        // 距离 = 地球半径 × 圆心角
        return EARTH_RADIUS * c
    }


}