package com.sd.cavphmi.net

import android.os.Handler
import android.os.Looper
import com.sd.cavphmi.moudule.NetworkModule
import okhttp3.ConnectionSpec
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.RequestBody
import okhttp3.sse.EventSource
import okhttp3.sse.EventSourceListener
import okhttp3.sse.EventSources
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.TimeUnit
import java.util.concurrent.locks.ReentrantLock
import kotlin.concurrent.withLock


/**
 * SSE 连接实体类：封装单个连接的核心信息
 * @param connId 连接唯一标识（如 "chat", "notice", "status"）
 * @param sseUrl 连接的服务器地址
 * @param eventSource SSE 连接实例（OkHttp 提供）
 * @param callback 连接的回调接口（接收消息、状态）
 */
data class SseConnection(
    val connId: String,
    val sseUrl: String,
    val eventSource: EventSource,
    val callback: SseCallback
)

/**
 * SSE 回调接口：每个连接的独立回调
 */
interface SseCallback {
    // 连接成功
    fun onConnected(connId: String)

    // 接收自定义事件（服务器指定 event 字段）
    fun onEventReceived(connId: String, eventType: String?, data: String)

    // 连接失败
    fun onFailed(connId: String, errorMsg: String)

    // 连接关闭
    fun onClosed(connId: String)
    fun onReconnecting(delayMillis: Long) // 重连中（可选）
}

class SseMultiConnectionManager {

    companion object {
        // 是否主动断开（控制重连逻辑）
        private var isManualDisconnect = false
        val instance: SseMultiConnectionManager by lazy { SseMultiConnectionManager() }
    }


    var sslData = NetworkModule.getSSlSocketFactory()

    // OkHttp 客户端（全局复用，避免重复创建；配置长连接适配 SSE）
    private val okHttpClient = OkHttpClient.Builder()
        .connectTimeout(30, TimeUnit.SECONDS)
        .readTimeout(0, TimeUnit.SECONDS) // 长连接延长读取超时
        .writeTimeout(30, TimeUnit.SECONDS)
//        .connectionPool(ConnectionPool(5, 30, TimeUnit.SECONDS)) // 连接池优化（可选）
//        .pingInterval(20, TimeUnit.SECONDS) // TCP心跳间隔
        .retryOnConnectionFailure(false) // 多连接场景下禁用自动重试（避免冲突）
        .connectionSpecs(
            listOf(
                ConnectionSpec.CLEARTEXT,
                ConnectionSpec.MODERN_TLS
            )
        ) // 支持 HTTP/HTTPS
        .sslSocketFactory(sslData.socketFactory, sslData.trustAllCert)
        .hostnameVerifier { hostname, session -> true }
        .build()

    // 存储所有活跃连接：key = connId（唯一标识），value = SseConnection
    // 使用 ConcurrentHashMap 保证线程安全（支持并发读写）
    private val activeConnections = ConcurrentHashMap<String, SseConnection>()

    // 重入锁：确保启动/关闭连接时的原子操作（避免并发问题）
    private val connectionLock = ReentrantLock()

    // 重连延迟（指数退避：1s → 2s → 4s → ... → 30s 上限）
    private var retryDelayMillis = 1000L
    private val maxRetryDelay = 30 * 1000L // 最大重连延迟

    /**
     * 启动一个新的 SSE 连接
     * @param connId 连接唯一标识（如 "chat_123", "system_notice"）
     * @param sseUrl 连接的服务器地址
     * @param callback 该连接的独立回调（处理消息和状态）
     */
    fun startConnection(
        connId: String,
        sseUrl: String,
        body: RequestBody,
        callback: SseCallback
    ) {
        connectionLock.withLock { // 加锁确保原子操作
            // 1. 先关闭同名连接（避免重复创建）
            if (activeConnections.containsKey(connId)) {
                stopConnection(connId)
                callback.onClosed(connId)
            }

            // 2. 构建当前连接的 SSE 请求（必须是 GET + text/event-stream 头）
            val request = Request.Builder()
                .url(sseUrl)
                .post(body)
                .header("Accept", "text/event-stream") // 核心：SSE 格式标识
                .header("Cache-Control", "no-cache")  // 禁用缓存
                .header("Connection", "keep-alive")   // 保持长连接
                // 可选：添加当前连接的独立头（如不同 Token、用户ID）
                // .header("Authorization", "Bearer ${getTokenForConn(connId)}")
                .build()

            // 3. 创建当前连接的 EventSourceListener（绑定回调）
            val listener = object : EventSourceListener() {
                override fun onOpen(eventSource: EventSource, response: okhttp3.Response) {
                    super.onOpen(eventSource, response)
//                    println("-------hashCode = ${eventSource.hashCode()}  ${eventSource.request().url}")
                    // 回调到当前连接的 callback（主线程）
                    callback.onConnected(connId)
                }


                override fun onEvent(
                    eventSource: EventSource,
                    id: String?,
                    type: String?,
                    data: String
                ) {
                    super.onEvent(eventSource, id, type, data)
                    // 自定义事件回调（服务器指定 event 字段）
                    callback.onEventReceived(connId, type, data)
                }

                override fun onFailure(
                    eventSource: EventSource,
                    t: Throwable?,
                    response: okhttp3.Response?
                ) {
                    super.onFailure(eventSource, t, response)
                    // 失败后移除连接（避免内存泄漏）
                    activeConnections.remove(connId)
                    // 失败回调（错误信息拼接）
                    val errorMsg = t?.message ?: "Unknown error"
                    callback.onFailed(connId, errorMsg)
                    // 主动断开时不自动重连（通过 flag 控制）
                    if (!isManualDisconnect) {
                        scheduleReconnect(connId, sseUrl, body, callback) // 被动断开则重连
                    }
                }

                override fun onClosed(eventSource: EventSource) {
                    super.onClosed(eventSource)
                    callback.onClosed(connId)
                    // 关闭后移除连接
                    activeConnections.remove(connId)
                }
            }

            // 4. 创建 EventSource 实例并存储到集合
            val eventSource = EventSources.createFactory(okHttpClient)
                .newEventSource(request, listener)

            // 5. 存储连接信息到 ConcurrentHashMap
            activeConnections[connId] = SseConnection(
                connId = connId,
                sseUrl = sseUrl,
                eventSource = eventSource,
                callback = callback
            )
        }
    }

    // 调度重连（指数退避策略）
    private fun scheduleReconnect(
        connId: String,
        sseUrl: String,
        body: RequestBody,
        sseCallback:SseCallback
    ) {
        sseCallback.onReconnecting(retryDelayMillis)
        // 使用 Handler 延迟重连（避免主线程阻塞）
        Handler(Looper.getMainLooper()).postDelayed({
            if (!isManualDisconnect) {
                startConnection(connId,sseUrl, body,sseCallback) // 重连
                // 指数退避：延迟翻倍，不超过最大值
                retryDelayMillis = minOf(retryDelayMillis * 2, maxRetryDelay)
            }
        }, retryDelayMillis)
    }


    /**
     * 关闭指定 ID 的 SSE 连接
     * @param connId 连接唯一标识
     */
    fun stopConnection(connId: String) {
        isManualDisconnect = true
        connectionLock.withLock {
            val connection = activeConnections.remove(connId)
            connection?.eventSource?.cancel() // 关闭连接
        }
    }

    /**
     * 关闭所有 SSE 连接（如 App 退出时）
     */
    fun stopAllConnections() {
        isManualDisconnect = true
        connectionLock.withLock {
            activeConnections.values.forEach { it.eventSource.cancel() }
            activeConnections.clear()
        }
    }

    /**
     * 检查连接是否活跃
     * @param connId 连接唯一标识
     * @return true = 活跃，false = 已关闭/未创建
     */
    fun isConnectionActive(connId: String): Boolean {
        return activeConnections.containsKey(connId)
    }

    /**
     * 获取所有活跃连接的 ID
     */
    fun getActiveConnIds(): List<String> {
        return activeConnections.keys.toList()
    }
}
