package commons.gis

import org.locationtech.jts.geom.Coordinate
import org.locationtech.jts.geom.GeometryFactory
import org.locationtech.jts.geom.LineString
import org.locationtech.jts.geom.LinearRing
import org.locationtech.jts.geom.Point
import org.locationtech.jts.geom.Polygon


object MGeoTools {

    // 创建 GeometryFactory
    private var geometryFactory: GeometryFactory = GeometryFactory()

    /**
     * 创建点对象
     * @param x 经度
     */
    fun createPoint(x: Double, y: Double): Point {
        val coord = Coordinate(x, y)
        return geometryFactory.createPoint(coord)
    }

    /**
     * 生成线段
     * @param points 经纬度线段
     */
    fun createLine(points: List<List<Double>>): LineString {
        val lines = points.map {
            Coordinate(it[0], it[1])
        }.toTypedArray()
        var lineString = geometryFactory.createLineString(lines)
        return lineString
    }
    
    /**
     * 创建 LinearRing
     * @param points 经纬度线段
     */
    fun createLinearRing(points: List<List<Double>>): LinearRing {
        val lines = points.map {
            Coordinate(it[0], it[1])
        }.toTypedArray()
        var linearRing = geometryFactory.createLinearRing(lines)
        return linearRing
    }

    // 创建多边形（无孔洞）
    fun createPolygonFromLinearRing(linearRing: LinearRing): Polygon {
        return geometryFactory.createPolygon(linearRing, null)
    }

    /***根据坐标生成前方矩形***/
    fun createRectangleFont(
        lng: Double, lat: Double, heading: Double, width: Double = 3.0, length: Double = 10.0,
    ): Polygon {
        var rectanglePoints =
            GeoRectangleUtils.createRectangleInFront(listOf(lng, lat), heading, width, length)

        val coordinates = rectanglePoints.map {
            Coordinate(it[0], it[1])
        }.toMutableList()
        var polygon = geometryFactory.createPolygon(coordinates.toTypedArray())
        return polygon
    }

    /***根据中心点生成指定方向的矩形**/
    fun createRectangleCenter(
        lng: Double, lat: Double, heading: Double, halfWidth: Double = 2.0,
        halfHeight: Double = 5.0
    ): Polygon {
        var rectanglePoints =
            GeoRectangleUtils.generateCenterRect(lng, lat, heading, halfWidth, halfHeight)

        val coordinates = rectanglePoints.map {
            Coordinate(it[0], it[1])
        }.toMutableList()
//        var one = myRectangle.corners.first()
//        coordinates.add(Coordinate(one[0], one[1]))
        var polygon = geometryFactory.createPolygon(coordinates.toTypedArray())
        return polygon
    }

    /**
     * 计算线段里的点是否在面里
     * @return 返回线段里点的下标
     */
    fun lineContainsArea(polygon: Polygon?, lineString: LineString?): Int {
        if (polygon == null || lineString == null) {
            return 0
        }
        lineString.coordinates.forEachIndexed { index, cood ->
            val sinPoint = geometryFactory.createPoint(cood)
            if (polygon.contains(sinPoint)) {
                return index
            }
        }
        return 0

    }



}