package com.cusc.map.utils;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.animation.ValueAnimator;
import android.view.animation.LinearInterpolator;

import com.minedata.minenavi.mapdal.LatLng;

public class SmoothMoveUtils {

    public interface OnPositionUpdateListener {
        /**
         * 每帧回调：插值点 + 朝向角度 bearing
         */
        void onUpdate(LatLng interpolatedLatLng, float bearing);

        /**
         * 动画结束
         */
        void onFinish();
    }

    public static void startSmoothMove(
            LatLng from,
            LatLng to,
            long durationMillis,
            OnPositionUpdateListener listener
    ) {
        if (from == null || to == null || listener == null || durationMillis <= 0) {
            return;
        }

        ValueAnimator animator = ValueAnimator.ofFloat(0f, 1f);
        animator.setDuration(durationMillis);
        animator.setInterpolator(new LinearInterpolator());

//        float bearing = calculateBearing(from, to);

        animator.addUpdateListener(animation -> {
            float fraction = (float) animation.getAnimatedValue();

            double lat = from.latitude + (to.latitude - from.latitude) * fraction;
            double lng = from.longitude + (to.longitude - from.longitude) * fraction;

            LatLng interpolated = new LatLng(lat, lng);
            listener.onUpdate(interpolated, 0); // 实时位置 + 朝向角
        });

        animator.addListener(new AnimatorListenerAdapter() {
            @Override
            public void onAnimationEnd(Animator animation) {
                listener.onFinish();
            }
        });

        animator.start();
    }

    /**
     * 计算从起点 to 终点的方位角（bearing），单位为度（0~360°）
     */
    private static float calculateBearing(LatLng from, LatLng to) {
        double lat1 = Math.toRadians(from.latitude);
        double lon1 = Math.toRadians(from.longitude);
        double lat2 = Math.toRadians(to.latitude);
        double lon2 = Math.toRadians(to.longitude);

        double dLon = lon2 - lon1;

        double y = Math.sin(dLon) * Math.cos(lat2);
        double x = Math.cos(lat1) * Math.sin(lat2) -
                   Math.sin(lat1) * Math.cos(lat2) * Math.cos(dLon);
        double bearingRad = Math.atan2(y, x);

        return (float) ((Math.toDegrees(bearingRad) + 360) % 360);
    }
}
