package com.cusc.map.maps

import android.location.Location
import com.amap.api.maps.AMapUtils
import com.amap.api.maps.model.LatLng
import com.cusc.map.MAP_TYPE
import com.cusc.map.MSDKInitializer
import com.cusc.map.maps.cdata.MSLatLng
import kotlin.math.atan2
import kotlin.math.cos
import kotlin.math.sin

/**
 * 地图计算工具
 */
object MSCalcuMapUtil {

    /**
     * 计算两点之间的直线距离
     * @param latLng1 第一个坐标点
     * @param latLng2 第二个坐标点
     * @return 返回两点间距离，单位为米
     */
    fun calculateLineDistance(
        latLng1: MSLatLng,
        latLng2: MSLatLng
    ): Float {
        // 根据不同的地图类型选择相应的距离计算方法
        when (MSDKInitializer.getMapType()) {
            MAP_TYPE.MINE -> {
                // 使用Android原生Location类计算两点间距离
                var results = FloatArray(1)
                Location.distanceBetween(
                    latLng1.lat,
                    latLng1.lng,
                    latLng2.lat,
                    latLng2.lng,
                    results
                )
                return results[0]
            }

            MAP_TYPE.AMAP -> {
                // 使用高德地图工具类计算两点间距离
                val latlngA = LatLng(latLng1.lat, latLng1.lng)
                val latlngB = LatLng(latLng2.lat, latLng2.lng)
                val distance = AMapUtils.calculateLineDistance(latlngA, latlngB)
                return distance
            }
        }
    }


    /**
     * 计算矩形面积
     * @param latLng1 矩形左上角坐标
     * @param latLng2 矩形右下角坐标
     * @return 矩形面积，单位为平方米
     */
    fun calculateArea(
        latLng1: MSLatLng,
        latLng2: MSLatLng
    ): Float {
        // 根据不同地图类型进行面积计算
        when (MSDKInitializer.getMapType()) {
            MAP_TYPE.MINE -> {
                return 0f
            }

            MAP_TYPE.AMAP -> {
                val leftTopLatlng = LatLng(latLng1.lat, latLng1.lng)
                val rightBottomLatlng = LatLng(latLng2.lat, latLng2.lng)
                val area = AMapUtils.calculateArea(leftTopLatlng, rightBottomLatlng)
                return area
            }
        }
    }


    /**
     * 根据两个坐标点计算航向角
     * @param fromPoint 起始点
     * @param toPoint 终点
     * @return 航向角（度），范围 0-360
     */
    fun calculateBearing(
        fromPoint: MSLatLng,
        toPoint: MSLatLng
    ): Double {
        return calculateBearing(
            fromPoint.lng, fromPoint.lat,
            toPoint.lng, toPoint.lat
        )
    }


    /**
     * 根据两个坐标点计算航向角
     * @param fromLon 起始点经度
     * @param fromLat 起始点纬度
     * @param toLon 终点经度
     * @param toLat 终点纬度
     * @return 航向角（度），范围 0-360
     */
   private fun calculateBearing(fromLon: Double, fromLat: Double, toLon: Double, toLat: Double): Double {
        // 将度转换为弧度
        val lat1 = Math.toRadians(fromLat)
        val lat2 = Math.toRadians(toLat)
        val deltaLon = Math.toRadians(toLon - fromLon)

        // 计算航向角
        val y = sin(deltaLon) * cos(lat2)
        val x = cos(lat1) * sin(lat2) - sin(lat1) * cos(lat2) * cos(deltaLon)
        val bearing = Math.toDegrees(atan2(y, x))

        // 确保航向角在 0-360 度范围内
        return (bearing + 360) % 360
    }


}