package com.sd.api.factorys

import androidx.core.graphics.toColorInt
import com.sd.api.maps.cdata.MSLatLng
import com.sd.api.bean.WayPoi
import com.sd.api.core.MapReadyView
import com.sd.api.maps.MSLocStyle
import com.sd.api.maps.cdata.MSMarker
import com.sd.api.maps.cdata.MSOverlay
import com.sd.api.core.basic.OnDriveRoute
import com.sd.api.maps.inters.OnMSPoiSearchLis
import com.sd.api.core.basic.OnMsGpsLoc
import com.sd.api.intfaces.OnNaviPresenterListener

/**
 * 基础接口
 */
interface IMapBasic {

    /**
     * 切换地图图层
     * @param mapReadView 地图准备就绪视图对象
     * @param type 1=普通 2=卫星
     */
    fun toggleLayers(mapReadView: MapReadyView?, type: Int)

    /**
     * 设置定位蓝点样式
     * @param mapReadView 地图准备就绪视图对象
     * @param type 定位类型，默认为LOCATION_TYPE_LOCATION_ROTATE，表示连续定位并旋转视角
     */
    fun setLocationStyle(
        mapReadView: MapReadyView?,
        type: Int = MSLocStyle.LOCATION_TYPE_LOCATION_ROTATE
    )

    /**
     * 设置地图旋转手势是否可用
     * @param mapReadView 地图准备就绪视图对象
     * @param enable 是否启用旋转手势功能
     */
    fun setRotateGesturesEnabled(mapReadView: MapReadyView?, enable: Boolean)


    /**
     * 设置地图倾斜手势功能的启用状态
     * @param mapReadView 地图准备就绪视图对象
     * @param enable 是否启用倾斜手势功能，true为启用，false为禁用
     * */
    fun setTiltGesturesEnabled(mapReadView: MapReadyView?, enable: Boolean)

    /**
     * 改变地图中心点
     * @param mapReadView 地图准备就绪的视图对象
     * @param lat 纬度
     * @param lng 经度
     * **/
    fun setMapCenter(
        mapReadView: MapReadyView?,
        lat: Double,
        lng: Double
    )


    /**
     * 改变地图缩放级别
     * @param mapReadView 地图准备就绪的视图对象
     * @param zoom 缩放级别，值越小站得越高，默认值为11f
     * */
    fun setMapZoom(mapReadView: MapReadyView?, zoom: Float = 11f)

    /**
     * 绘制 marker 点
     * @param mapReadView 地图准备就绪的视图对象
     * @param lat 纬度坐标
     * @param lng 经度坐标
     * @return MSMarker 标记点对象
     */
    fun drawMarket(
        mapReadView: MapReadyView?,
        lat: Double,
        lng: Double
    ): MSMarker

    /**
     * 绘制线
     * @param mapReadView 地图加载返回
     * @param msLatLng 坐标点集合
     * @param width 线宽度，默认为10f
     * @param fcolor 线颜色
     * @param isDotted 是否为虚线，默认为false
     * @return MSOverlay 覆盖物对象
     */
    fun drawPolyline(
        mapReadView: MapReadyView?,
        msLatLng: List<MSLatLng>,
        fcolor: Int,
        width: Float,
        isDotted: Boolean = false
    ): MSOverlay


    /**绘制圆形
     * @param mapReadView 地图准备就绪视图对象
     * @param lat 维度
     * @param lng 经度
     * @param radius 半径（米）
     * @param fillColor 填充颜色，默认为半透明红色
     * @param strokeColor 边框颜色，默认为半透明蓝色
     * @param strokeWidth 边框宽度，默认为15f
     * @return MSOverlay 返回覆盖物对象
     */
    fun drawCircle(
        mapReadView: MapReadyView?,
        lat: Double,
        lng: Double,
        radius: Float,
        fillColor: Int = "#50FF0000".toColorInt(),
        strokeColor: Int = "#500000FF".toColorInt(),
        strokeWidth: Float = 15f
    ): MSOverlay


    /***
     * 绘制多边形
     * @param msLatLng   添加多边形顶点坐标集合
     * @param dLine 是否虚线
     * @param fillColor 填充颜色
     * @param strokeWidth 边框宽度
     * @param strokeColor 边框颜色
     * ****/
    fun drawPolygon(
        mapReadView: MapReadyView?,
        msLatLng: List<MSLatLng>,
        dLine: Boolean = false,
        fillColor: Int = "#10FF00FF".toColorInt(),
        strokeWidth: Float = 5f,
        strokeColor: Int = "#50FF00FF".toColorInt()
    ): MSOverlay


    /**关键字搜索
     * @param keyWord 关键字
     * @param lat 当期维度
     * @param lng 当期经度
     * @param onMSPoiSearchLis 搜索回调
     */
    fun startPoiSearch(
        keyWord: String,
        lat: Double,
        lng: Double,
        onMSPoiSearchLis: OnMSPoiSearchLis
    )

    /**周边搜索
     * @param keyWord 关键字
     * @param lat 当前维度
     * @param lng 当前经度
     * @param radios 搜索半径(单位 米)
     * @param onMSPoiSearchLis 回调
     */
    fun startPoiSearchBound(
        keyWord: String,
        lat: Double,
        lng: Double,
        radios: Int,
        onMSPoiSearchLis: OnMSPoiSearchLis
    )

    /**
     * 开启定位
     * 设置GPS定位回调监听器
     * @param onMsGpsLoc GPS定位回调监听器实例
     */
    fun startLoc(onMsGpsLoc: OnMsGpsLoc)

    /**停止定位*/
    fun stopLoc()

    /**清理定位资源**/
    fun cleansLoc()

    /**驾车路径规划
     * @param starPoint 起点坐标
     * @param startName 起点名称
     * @param endPoint 终点坐标
     * @param endName 终点名称（高德可不传）
     * @param ways     途经点集合
     * @param mapReadView 地图对象
     * @param onDriveRoute 路径规划结果回调
     */
    fun drivingPathPlanning(
        starPoint: MSLatLng,
        startName: String = "当前位置",
        endPoint: MSLatLng,
        endName: String = "",
        ways: List<WayPoi>,
        mapReadView: MapReadyView?,
        onDriveRoute: OnDriveRoute?
    )

    /**
     * 开始导航
     * @param navType 1=自己实现回调 2=直接调用导航组件
     * @param starPoint 起点坐标
     * @param startName 起点名称
     * @param endPoint 终点坐标
     * @param endName 终点名称
     * @param ways 途经点集合
     * @param onNaviPresenterListener 四维导航监听回调
     */
    fun startNavi(
        mapReadView: MapReadyView?,
        navType: Int,
        starPoint: MSLatLng,
        startName: String = "当前位置",
        endPoint: MSLatLng,
        endName: String = "",
        ways: List<WayPoi>,
        onNaviPresenterListener:OnNaviPresenterListener?
    )


}