package com.sd.api.maps.amap

import android.graphics.BitmapFactory
import androidx.core.graphics.toColorInt
import com.amap.api.maps.model.BitmapDescriptorFactory
import com.amap.api.maps.model.CircleOptions
import com.amap.api.maps.model.LatLng
import com.amap.api.maps.model.MarkerOptions
import com.amap.api.maps.model.PolygonOptions
import com.amap.api.maps.model.PolylineOptions
import com.msmap.api.R
import com.sd.api.UserCtx
import com.sd.api.maps.cdata.MSLatLng
import com.sd.api.ui.MapReadyView
import com.sd.api.maps.cdata.MSOverlay
import com.sd.api.maps.cdata.MSMarker

/**
 * 在地图上绘制
 */
object AmapDrawInMap {


    /**
     * 绘制 marker 点
     * @param mapReadView 地图准备就绪的视图对象
     * @param lat 纬度坐标
     * @param lng 经度坐标
     * @return MSMarker 标记点对象
     */
    fun drawMarket(
        mapReadView: MapReadyView?,
        lat: Double,
        lng: Double
    ): MSMarker {
        var context = UserCtx.Companion.instance.mContext

        // 处理高德地图类型的标记点绘制
        val aLatLng = LatLng(lat, lng)
        val aMarkerOption = MarkerOptions().apply {
            position(aLatLng)
            icon(
                BitmapDescriptorFactory.fromBitmap(
                    BitmapFactory
                        .decodeResource(context?.resources, R.drawable.market_loc)
                )
            )
        }
        var aMaker = mapReadView?.aMap?.addMarker(aMarkerOption)
        MSMarker.amaker = aMaker
        return MSMarker
    }


    /**
     * 绘制线
     * @param mapReadView 地图加载返回
     * @param msLatLng 坐标点集合
     * @param width 线宽度，默认为10f
     * @param fcolor 线颜色，默认为"#FF00FF"对应的int值
     * @param isDotted 是否为虚线，默认为false
     * @return MSOverlay 覆盖物对象
     */
    fun drawPolyline(
        mapReadView: MapReadyView?,
        msLatLng: List<MSLatLng>,
        fcolor: Int,
        width: Float,
        isDotted: Boolean = false
    ): MSOverlay {
        // 绘制高德地图的线
        var latLngs = msLatLng.map {
            LatLng(it.lat, it.lng)
        }
        val aOption = PolylineOptions().addAll(latLngs)
            .width(width)
            .color(fcolor)
            .setDottedLine(isDotted)
        var polyline = mapReadView?.aMap?.addPolyline(aOption)
        MSOverlay.aPolyline = polyline
        return MSOverlay
    }


    /**绘制圆形
     * @param mapReadView 地图准备就绪视图对象
     * @param lat 维度
     * @param lng 经度
     * @param radius 半径（米）
     * @param fillColor 填充颜色，默认为半透明红色
     * @param strokeColor 边框颜色，默认为半透明蓝色
     * @param strokeWidth 边框宽度，默认为15f
     * @return MSOverlay 返回覆盖物对象
     */
    fun drawCircle(
        mapReadView: MapReadyView?,
        lat: Double,
        lng: Double,
        radius: Float,
        fillColor: Int = "#50FF0000".toColorInt(),
        strokeColor: Int = "#500000FF".toColorInt(),
        strokeWidth: Float = 15f
    ): MSOverlay {
        var latLng = LatLng(lat, lng)
        val options = CircleOptions()
            .center(latLng)
            .radius(radius.toDouble())
            .fillColor(fillColor)
            .strokeColor(strokeColor)
            .strokeWidth(strokeWidth)
        var circle = mapReadView?.aMap?.addCircle(options)
        MSOverlay.aCircle = circle
        return MSOverlay
    }


    /***
     * 绘制多边形
     * @param msLatLng   添加多边形顶点坐标集合
     * @param dLine 是否虚线
     * @param fillColor 填充颜色
     * @param strokeWidth 边框宽度
     * @param strokeColor 边框颜色
     * ****/
    fun drawPolygon(
        mapReadView: MapReadyView?,
        msLatLng: List<MSLatLng>,
        dLine: Boolean = false,
        fillColor: Int = "#10FF00FF".toColorInt(),
        strokeWidth: Float = 5f,
        strokeColor: Int = "#50FF00FF".toColorInt()
    ): MSOverlay {
        var latLngs = msLatLng.map {
            LatLng(it.lat, it.lng)
        }
        val aPolygonOptions = PolygonOptions().apply {
            addAll(latLngs)
            fillColor(fillColor)
            strokeWidth(strokeWidth)
            strokeColor(strokeColor)
        }
        var polygon = mapReadView?.aMap?.addPolygon(aPolygonOptions)
        MSOverlay.aPolygon = polygon
        return MSOverlay

    }

}