package com.sd.api.maps

import android.location.Location
import com.amap.api.maps.AMapUtils
import com.amap.api.maps.model.LatLng
import com.sd.api.MAP_TYPE
import com.sd.api.MSDKInitializer
import com.sd.api.maps.cdata.MSLatLng

/**
 * 地图计算工具
 */
object MSCalcuMapUtil {

    /**
     * 计算两点之间的直线距离
     * @param latLng1 第一个坐标点
     * @param latLng2 第二个坐标点
     * @return 返回两点间距离，单位为米
     */
    fun calculateLineDistance(
        latLng1: MSLatLng,
        latLng2: MSLatLng
    ): Float {
        // 根据不同的地图类型选择相应的距离计算方法
        when (MSDKInitializer.getMapType()) {
            MAP_TYPE.MINE -> {
                // 使用Android原生Location类计算两点间距离
                var results = FloatArray(1)
                Location.distanceBetween(
                    latLng1.lat,
                    latLng1.lng,
                    latLng2.lat,
                    latLng2.lng,
                    results
                )
                return results[0]
            }

            MAP_TYPE.AMAP -> {
                // 使用高德地图工具类计算两点间距离
                val latlngA = LatLng(latLng1.lat, latLng1.lng)
                val latlngB = LatLng(latLng2.lat, latLng2.lng)
                val distance = AMapUtils.calculateLineDistance(latlngA, latlngB)
                return distance
            }
        }
    }


    /**
     * 计算矩形面积
     * @param latLng1 矩形左上角坐标
     * @param latLng2 矩形右下角坐标
     * @return 矩形面积，单位为平方米
     */
    fun calculateArea(
        latLng1: MSLatLng,
        latLng2: MSLatLng
    ): Float {
        // 根据不同地图类型进行面积计算
        when (MSDKInitializer.getMapType()) {
            MAP_TYPE.MINE -> {
                return 0f
            }

            MAP_TYPE.AMAP -> {
                val leftTopLatlng = LatLng(latLng1.lat, latLng1.lng)
                val rightBottomLatlng = LatLng(latLng2.lat, latLng2.lng)
                val area = AMapUtils.calculateArea(leftTopLatlng, rightBottomLatlng)
                return area
            }
        }
    }

}