package com.sd.api

import android.Manifest
import android.content.Context
import android.os.Build
import androidx.fragment.app.FragmentActivity
import com.amap.api.maps.MapsInitializer
import com.minedata.minenavi.SDKInitializer
import com.minedata.minenavi.SDKInitializer.InitListener
import com.minedata.minenavi.mapdal.CoordType
import com.minedata.minenavi.poiquery.SearchUrlType
import com.permissionx.guolindev.PermissionX


/**
 * AMAP 高德在线地图
 * MINE 四维在线地图
 */
enum class MAP_TYPE {
    MINE,
    AMAP
}

/**用户传入上下文*/
class UserCtx {
    var mContext: Context? = null
    var map_type: MAP_TYPE = MAP_TYPE.MINE

    companion object {
        val instance: UserCtx by lazy { UserCtx() }
    }
}


interface OnSdkInitCb {
    /**初始化成功*/
    fun onInitSuccess()

    /**初始化成失败*/
    fun onInitFailed(msg: String)
}

/**SDK初始化类*/
object MSDKInitializer {

    /**SDK初始化成功（失败）回调*/
    private var onSdkInitCb: OnSdkInitCb? = null

    /**
     * 初始化引擎
     *
     * @param context 应用上下文，用于SDK初始化
     * @param map_type 地图类型，指定要初始化的地图类型
     * @param onSdkInitCb SDK初始化回调接口，用于通知初始化结果
     */
    fun initialize(context: Context, map_type: MAP_TYPE, onSdkInitCb: OnSdkInitCb) {
        // 保存初始化回调接口
        this.onSdkInitCb = onSdkInitCb
        // 设置用户上下文信息
        UserCtx.instance.mContext = context
        UserCtx.instance.map_type = map_type
        // 请求必要的权限
        requestPers()
    }

    /**
     * 获取当前用户的地图类型设置
     *
     * @return 返回当前用户设置的地图类型枚举值
     */
    fun getMapType(): MAP_TYPE {
        return UserCtx.instance.map_type
    }


    /**
     * 初始化地图类型相关配置
     */
    private fun mapTypeInit() {
        val ctx = UserCtx.instance.mContext
        when (UserCtx.instance.map_type) {
            MAP_TYPE.AMAP -> {
                // 高德地图隐私合规设置
                MapsInitializer.updatePrivacyShow(ctx, true, true)
                MapsInitializer.updatePrivacyAgree(ctx, true)
                // 高德地图不支持API初始化，直接回调失败
                onSdkInitCb?.onInitFailed("高德不支持api初始化")
            }

            MAP_TYPE.MINE -> {
                // 隐私合规接口
                SDKInitializer.setAgreePrivacy(true)
                //设置地图坐标系
                SDKInitializer.setCoordType(CoordType.GCJ02)
                SDKInitializer.setSearchUrlType(SearchUrlType.v1)
                // 执行SDK初始化
                SDKInitializer.initialize(ctx, object : InitListener {
                    override fun onInitSuccess() {
                        onSdkInitCb?.onInitSuccess()
                    }

                    override fun onInitFailed(msg: String) {
                        onSdkInitCb?.onInitFailed(msg)
                    }
                })
            }

            else -> {


            }
        }
    }

    // 请求权限
    private fun requestPers() {
        if (UserCtx.instance.mContext is FragmentActivity) {
            var pers = mutableListOf(
                Manifest.permission.ACCESS_COARSE_LOCATION,
                Manifest.permission.ACCESS_FINE_LOCATION,
                Manifest.permission.ACCESS_BACKGROUND_LOCATION,
                Manifest.permission.READ_PHONE_STATE
            )
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
//                pers.add(Manifest.permission.READ_EXTERNAL_STORAGE)
                pers.add(Manifest.permission.MANAGE_EXTERNAL_STORAGE)
            } else {
                pers.add(Manifest.permission.READ_EXTERNAL_STORAGE)
            }
            PermissionX.init(UserCtx.instance.mContext as FragmentActivity)
                .permissions(pers)
                .onExplainRequestReason { scope, deniedList ->
                    scope.showRequestReasonDialog(
                        deniedList,
                        "需要同意以下授权才能正常使用",
                        "好的",
                        "取消"
                    )
                }
//            .onForwardToSettings { scope, deniedList ->
//                scope.showForwardToSettingsDialog(deniedList, "您需要手动在‘设置’中允许必要的权限", "OK", "Cancel")
//            }
                .request { allGranted, grantedList, deniedList ->
                    if (allGranted) {
                        mapTypeInit()
                    } else {
                        onSdkInitCb?.onInitFailed("权限被拒")
                    }
                }

        } else {
//            ToastHelper.showShort(UserCtx.instance.mContext!!, "请在正确的地方初始化")
        }
    }


}