package com.sd.cavphmi.utils;

import android.graphics.Point;
import android.location.Location;

import com.minedata.minenavi.mapdal.LatLng;

import java.lang.reflect.Array;

public class MyMapUtils {


    /**
     * 根据两个坐标点计算航向角
     * @param fromPoint 起始点
     * @param toPoint 终点
     * @return 航向角（度），范围 0-360
     */
    public static double calculateBearing(LatLng fromPoint, LatLng toPoint) {
        return calculateBearing(
                fromPoint.longitude, fromPoint.latitude,
                toPoint.longitude, toPoint.latitude
        );
    }

    /**
     * 根据两个坐标点计算航向角
     * @param fromLon 起始点经度
     * @param fromLat 起始点纬度
     * @param toLon 终点经度
     * @param toLat 终点纬度
     * @return 航向角（度），范围 0-360
     */
    public static double calculateBearing(double fromLon, double fromLat, double toLon, double toLat) {
        // 将度转换为弧度
        double lat1 = Math.toRadians(fromLat);
        double lat2 = Math.toRadians(toLat);
        double deltaLon = Math.toRadians(toLon - fromLon);

        // 计算航向角
        double y = Math.sin(deltaLon) * Math.cos(lat2);
        double x = Math.cos(lat1) * Math.sin(lat2) - Math.sin(lat1) * Math.cos(lat2) * Math.cos(deltaLon);
        double bearing = Math.toDegrees(Math.atan2(y, x));

        // 确保航向角在 0-360 度范围内
        return (bearing + 360) % 360;
    }

    /**
     * 根据三个坐标点计算航向角
     * 通过计算第一点到第二点的航向角和第二点到第三点的航向角的平均值来得到更准确的航向角
     * @param point1 第一个点
     * @param point2 第二个点（当前位置）
     * @param point3 第三个点
     * @return 航向角（度），范围 0-360
     */
    public static double calculateBearing(LatLng point1, LatLng point2, LatLng point3) {
        // 计算第一点到第二点的航向角
        double bearing1 = calculateBearing(point1, point2);

        // 计算第二点到第三点的航向角
        double bearing2 = calculateBearing(point2, point3);

        // 返回两个航向角的平均值
        return averageBearing(bearing1, bearing2);
    }

    /**
     * 根据三个坐标点计算航向角
     * 通过计算第一点到第二点的航向角和第二点到第三点的航向角的平均值来得到更准确的航向角
     * @param lon1 第一点经度
     * @param lat1 第一点纬度
     * @param lon2 第二点经度
     * @param lat2 第二点纬度
     * @param lon3 第三点经度
     * @param lat3 第三点纬度
     * @return 航向角（度），范围 0-360
     */
    public static double calculateBearing(double lon1, double lat1, double lon2, double lat2, double lon3, double lat3) {
        // 计算第一点到第二点的航向角
        double bearing1 = calculateBearing(lon1, lat1, lon2, lat2);

        // 计算第二点到第三点的航向角
        double bearing2 = calculateBearing(lon2, lat2, lon3, lat3);

        // 返回两个航向角的平均值
        return averageBearing(bearing1, bearing2);
    }

    /**
     * 计算两个航向角的平均值，考虑0度和360度的边界情况
     * @param bearing1 航向角1
     * @param bearing2 航向角2
     * @return 平均航向角（度），范围 0-360
     */
    private static double averageBearing(double bearing1, double bearing2) {
        // 如果两个角度相差大于180度，需要特殊处理
        if (Math.abs(bearing1 - bearing2) > 180) {
            if (bearing1 > bearing2) {
                bearing2 += 360;
            } else {
                bearing1 += 360;
            }
        }

        // 计算平均值
        double average = (bearing1 + bearing2) / 2;

        // 确保结果在0-360度范围内
        return average % 360;
    }

    //计算两点距离
    public static double cauMyLocDistance(double fromLon, double fromLat, double toLon, double toLat) {
        var results = new float[4];
        Location.distanceBetween(
                fromLat,
                fromLon,
                toLat,
                toLon,
                results
        );
        float distanceInMeters = results[0];
        return distanceInMeters;
    }





}
