package cn.spatiotemporal.web.core.utils;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Year;
import java.time.YearMonth;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

/**
 * @ClassName: DateUtils
 * @Description: ThreadLocal线程安全的DateUtils
 * @date 2020年1月9日 上午10:43:42
 * 
 * @author Q.JI
 * @version
 * @since JDK 1.8
 */
public class DateUtils {
	private DateUtils() {
	}

	private static final ZoneId ZONE_ID = ZoneId.systemDefault();

	/**
	 * LocalDateTime转化为Date
	 * 
	 * @param localDateTime
	 * @return
	 */
	public static Date toDate(LocalDateTime localDateTime) {
		return Date.from(localDateTime.atZone(ZONE_ID).toInstant());
	}

	/**
	 * LocalDateTime转化为Date
	 * 
	 * @param localDateTime
	 * @return
	 */
	public static Date toDate(LocalDate localDate) {
		return Date.from(localDate.atStartOfDay(ZONE_ID).toInstant());
	}

	/**
	 * Date转化为LocalDateTime
	 * 
	 * @param date
	 * @return
	 */
	public static LocalDateTime toLocalDateTime(Date date) {
		return LocalDateTime.ofInstant(date.toInstant(), ZONE_ID);
	}

	/**
	 * LocalDate转化为LocalDateTime
	 * 
	 * @param localDate
	 * @return
	 */
	public static LocalDateTime toLocalDateTime(LocalDate localDate) {
		return LocalDateTime.of(localDate, LocalTime.MIN);
	}

	/**
	 * Date转化为LocalDate
	 * 
	 * @param date
	 * @return
	 */
	public static LocalDate toLocalDate(Date date) {
		return date.toInstant().atZone(ZONE_ID).toLocalDate();
	}

	/**
	 * Date转化为字符串
	 * 
	 * @param date
	 * @param formatter
	 * @return
	 */
	public static String format(Date date, DateFormatter formatter) {
		LocalDateTime localDateTime = LocalDateTime.ofInstant(date.toInstant(), ZONE_ID);
		return formatter.getDateTimeFormatter().format(localDateTime);
	}

	/**
	 * Date转化为字符串
	 * 
	 * @param date
	 * @param format
	 * @return
	 */
	public static String format(Date date, String format) {
		LocalDateTime localDateTime = LocalDateTime.ofInstant(date.toInstant(), ZONE_ID);
		return DateTimeFormatter.ofPattern(format, Locale.CHINA).format(localDateTime);
	}

	/**
	 * 字符串转化为Date
	 * 
	 * @param text
	 * @param formatter
	 * @return
	 */
	public static Date parse(String text, DateFormatter formatter) {
		return formatter.parse(text);
	}

	/**
	 * 
	 * @Title: increase
	 * @Description: 时间相加减运算
	 * @param date
	 * @param dateType
	 * @param amount
	 * @param formatter
	 * @return String
	 */
	public static String increase(Date date, int dateType, int amount, DateFormatter formatter) {
		Date myDate = null;
		if (date != null) {
			Calendar calendar = Calendar.getInstance();
			calendar.setTime(date);
			calendar.add(dateType, amount);
			myDate = calendar.getTime();
		}
		return format(myDate, formatter);
	}

	/**
	 * 
	 * @ClassName: DateFormatter
	 * @Description: 时间格式枚举
	 * @date 2020年1月17日 下午4:04:01
	 * 
	 * @author Q.JI
	 * @version
	 * @since JDK 1.8
	 */
	public static enum DateFormatter {
		/**
		 * 格式yyyy
		 */
		YEAR_FORMATTER(DateTimeFormatter.ofPattern("yyyy", Locale.CHINA)) {
			@Override
			public Date parse(String text) {
				Year year = Year.parse(text, dateTimeFormatter);
				return Date.from(year.atDay(1).atStartOfDay(ZONE_ID).toInstant());
			}
		},

		/**
		 * yyyy-MM
		 */
		YEAR_MONTH_FORMATTER(DateTimeFormatter.ofPattern("yyyy-MM", Locale.CHINA)) {
			@Override
			public Date parse(String text) {
				YearMonth yearMonth = YearMonth.parse(text, dateTimeFormatter);
				return Date.from(yearMonth.atDay(1).atStartOfDay(ZONE_ID).toInstant());
			}
		},

		/**
		 * 格式yyyy-MM-dd
		 */
		DATE_FORMATTER(DateTimeFormatter.ofPattern("yyyy-MM-dd", Locale.CHINA)) {
			@Override
			public Date parse(String text) {
				LocalDate localDate = LocalDate.parse(text, dateTimeFormatter);
				return Date.from(localDate.atStartOfDay(ZONE_ID).toInstant());
			}
		},

		/**
		 * 格式yyyy-MM-dd HH:mm:ss
		 */
		DATE_TIME_FORMATTER(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss", Locale.CHINA)) {
			@Override
			public Date parse(String text) {
				LocalDateTime localDateTime = LocalDateTime.parse(text, dateTimeFormatter);
				return Date.from(localDateTime.atZone(ZONE_ID).toInstant());
			}
		},

		/**
		 * 格式yyyy-MM-dd_HH
		 */
		DATE_HOUR_FORMATTER(DateTimeFormatter.ofPattern("yyyy-MM-dd_HH", Locale.CHINA)) {
			@Override
			public Date parse(String text) {
				LocalDate localDate = LocalDate.parse(text, dateTimeFormatter);
				return Date.from(localDate.atStartOfDay(ZONE_ID).toInstant());
			}
		},

		/**
		 * 格式yyyyMMdd
		 */
		YYYYMMDD_FORMATTER(DateTimeFormatter.ofPattern("yyyyMMdd", Locale.CHINA)) {
			@Override
			public Date parse(String text) {
				LocalDateTime localDateTime = LocalDateTime.parse(text, dateTimeFormatter);
				return Date.from(localDateTime.atZone(ZONE_ID).toInstant());
			}
		},

		/**
		 * 格式yyyyMMddHHmmss
		 */
		YYYYMMDDHHMMSS_FORMATTER(DateTimeFormatter.ofPattern("yyyyMMddHHmmss", Locale.CHINA)) {
			@Override
			public Date parse(String text) {
				LocalDateTime localDateTime = LocalDateTime.parse(text, dateTimeFormatter);
				return Date.from(localDateTime.atZone(ZONE_ID).toInstant());
			}
		},

		/**
		 * 格式yyyyMMdd_HHmmss
		 */
		YYYYMMDD_HHMMSS_FORMATTER(DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss", Locale.CHINA)) {
			@Override
			public Date parse(String text) {
				LocalDateTime localDateTime = LocalDateTime.parse(text, dateTimeFormatter);
				return Date.from(localDateTime.atZone(ZONE_ID).toInstant());
			}
		};

		protected DateTimeFormatter dateTimeFormatter;

		private DateFormatter(DateTimeFormatter dateTimeFormatter) {
			this.dateTimeFormatter = dateTimeFormatter;
		}

		public DateTimeFormatter getDateTimeFormatter() {
			return dateTimeFormatter;
		}

		public abstract Date parse(String text);
	}
}
